//  $Id: bioshook.cc,v 1.4 2001/07/13 15:15:06 nishi Exp $
//
//  Copyright (C) 2001 Shouhei Nishi.
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Lesser General Public
//  License as published by the Free Software Foundation; either
//  version 2 of the License, or (at your option) any later version.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public
//  License along with this library; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

#include "bochs.h"
#if BX_EMULATION_TOWNS
#define LOG_THIS BX_CPU_THIS_PTR

#if BX_USE_CPU_SMF
#define this (BX_CPU(0))
#endif

  void
BX_CPU_C::regist_bioshook(int type,
			  Bit32u addr,
			  Boolean isPM,
			  bios_trapfunc trapfunc,
			  void *this_ptr)
{
  int off;
  int i;

  switch(type) {
  case HOOK_RAM:
  case HOOK_ROM:
    switch(type) {
    case HOOK_RAM:
      off = isPM ? HOOK_OFF_RAM_PM : HOOK_OFF_RAM_RM;
      break;
    case HOOK_ROM:
    default:
      off = isPM ? HOOK_OFF_ROM_PM : HOOK_OFF_ROM_RM;
      break;
    }
    for(i=0;i<MAX_MEM_HOOK;i++) {
      if(BX_CPU_THIS_PTR memhook[i][off].enable &&
	 BX_CPU_THIS_PTR memhook[i][off].addr == addr) {
	goto already;
      }
    }
    for(i=0;i<MAX_MEM_HOOK;i++) {
      if(!BX_CPU_THIS_PTR memhook[i][off].enable) {
	BX_CPU_THIS_PTR memhook[i][off].enable = 1;
	BX_CPU_THIS_PTR memhook[i][off].addr = addr;
	BX_CPU_THIS_PTR memhook[i][off].trapfunc = trapfunc;
	BX_CPU_THIS_PTR memhook[i][off].this_ptr = this_ptr;
	switch(type) {
	case HOOK_RAM:
	  if(addr >= bx_mem.len) goto illhook;
	  bx_mem.vector[addr] = 0xcc;
	  return;
	case HOOK_ROM:
	  if(addr >= 0x40000) goto illhook;
	  bx_mem.sysromx[addr] = 0xcc;
	  return;
	}
	return;
      }
    }
    goto nofree;
    return;
  case HOOK_SINT:
  case HOOK_HINT:
  case HOOK_EXCEP:
  case HOOK_EXCEP_NO:
    if(addr>=0x100) {
      goto illhook;
    }
    switch(type) {
    case HOOK_SINT:
      off = isPM ? HOOK_OFF_SINT_PM : HOOK_OFF_SINT_RM;
      break;
    case HOOK_HINT:
      off = isPM ? HOOK_OFF_HINT_PM : HOOK_OFF_HINT_RM;
      break;
    case HOOK_EXCEP:
    case HOOK_EXCEP_NO:
    default:
      off = isPM ? HOOK_OFF_EXCEP_PM : HOOK_OFF_EXCEP_RM;
      break;
    }
    if(BX_CPU_THIS_PTR inthook[addr][off].enable) {
      goto already;
    }
    BX_CPU_THIS_PTR inthook[addr][off].enable = 1;
    BX_CPU_THIS_PTR inthook[addr][off].trapfunc = trapfunc;
    BX_CPU_THIS_PTR inthook[addr][off].this_ptr = this_ptr;
    return;
  default:
  illhook:
    BX_PANIC(("cpu:illegal bios hook addr=%.8x, type=%d, isPM=%d.\n",
	      addr,
	      type,
	      isPM));
  }
already:
  BX_PANIC(("cpu:bios hook already installed to addr=%.8x, type=%d, isPM=%d.\n",
	    addr,
	    type,
	    isPM));
nofree:
  BX_PANIC(("cpu:too more bios hook type=%d.\n",
	    type));
}

  void
BX_CPU_C::remove_bioshook(int type,
			  Bit32u addr,
			  Boolean isPM)
{
  int off;
  int i;

  switch(type) {
  case HOOK_RAM:
  case HOOK_ROM:
    switch(type) {
    case HOOK_RAM:
      off = isPM ? HOOK_OFF_RAM_PM : HOOK_OFF_RAM_RM;
      break;
    case HOOK_ROM:
    default:
      off = isPM ? HOOK_OFF_ROM_PM : HOOK_OFF_ROM_RM;
      break;
    }
    for(i=0;i<MAX_MEM_HOOK;i++) {
      if(BX_CPU_THIS_PTR memhook[i][off].enable &&
	 BX_CPU_THIS_PTR memhook[i][off].addr == addr) {
	BX_CPU_THIS_PTR memhook[i][off].enable = 0;
	switch(type) {
	case HOOK_ROM:
	  if(addr >= 0x40000) goto illhook;
	  bx_mem.sysromx[addr] = bx_mem.sysrom[addr];
	  return;
	}
	return;
      }
    }
    goto notinstalled;
    return;
  case HOOK_SINT:
  case HOOK_HINT:
  case HOOK_EXCEP:
  case HOOK_EXCEP_NO:
    if(addr>=0x100) {
      BX_PANIC(("cpu:illegal bios hook addr=%.8x, type=%d.\n",
		addr,
		type));
    }
    switch(type) {
    case HOOK_SINT:
      off = isPM ? HOOK_OFF_SINT_PM : HOOK_OFF_SINT_RM;
      break;
    case HOOK_HINT:
      off = isPM ? HOOK_OFF_HINT_PM : HOOK_OFF_HINT_RM;
      break;
    case HOOK_EXCEP:
    case HOOK_EXCEP_NO:
    default:
      off = isPM ? HOOK_OFF_EXCEP_PM : HOOK_OFF_EXCEP_RM;
      break;
    }
    if(!BX_CPU_THIS_PTR inthook[addr][off].enable) {
      goto notinstalled;
    }
    BX_CPU_THIS_PTR inthook[addr][off].enable = 0;
    return;
  default:
  illhook:
    BX_PANIC(("cpu:illegal bios hook addr=%.8x, type=%d, isPM=%d.\n",
	      addr,
	      type,
	      isPM));
  }
notinstalled:
  BX_PANIC(("cpu:bios hook not installed to addr=%.8x, type=%d, isPM=%d.\n",
	    addr,
	    type,
	    isPM));
}

  void
BX_CPU_C::init_bioshook(void)
{
  int i,j;

  for(i=0;i<MAX_MEM_HOOK;i++) {
    for(j=0;j<4;j++) {
      BX_CPU_THIS_PTR memhook[i][j].enable = 0;
    }
  }
  for(i=0;i<0x100;i++) {
    for(j=0;j<6;j++) {
      BX_CPU_THIS_PTR inthook[i][j].enable = 0;
    }
  }

  memcpy(bx_mem.sysromx,bx_mem.sysrom,0x40000);
}

  Boolean
BX_CPU_C::exec_bioshook(int type,
			Bit32u addr,
			Bit32u errorno)
{
  int off;
  int i;
  Boolean isPM = protected_mode();
  bios_trapfunc trapfunc = NULL;
  void *this_ptr = NULL;
  int exittype;
  Bit32u exitaddr;
  Bit16u exitseg;
  Bit64u leasttime,maxtime;
  Bit64u time,start_time;
  BxInstruction_t ix;

  switch(type) {
  case HOOK_RAM:
  case HOOK_ROM:
    exittype = HOOK_END_FRET;
    switch(type) {
    case HOOK_RAM:
      off = isPM ? HOOK_OFF_RAM_PM : HOOK_OFF_RAM_RM;
      break;
    case HOOK_ROM:
    default:
      off = isPM ? HOOK_OFF_ROM_PM : HOOK_OFF_ROM_RM;
      break;
    }
    for(i=0;i<MAX_MEM_HOOK;i++) {
      if(BX_CPU_THIS_PTR memhook[i][off].enable &&
	 BX_CPU_THIS_PTR memhook[i][off].addr == addr) {
	trapfunc = BX_CPU_THIS_PTR memhook[i][off].trapfunc;
	this_ptr = BX_CPU_THIS_PTR memhook[i][off].this_ptr;
	goto found;
      }
    }
    return 0;
  case HOOK_SINT:
  case HOOK_HINT:
  case HOOK_EXCEP:
  case HOOK_EXCEP_NO:
    if(addr>=0x100) {
      goto illhook;
    }
    exittype = HOOK_END_DONE;
    switch(type) {
    case HOOK_SINT:
      off = isPM ? HOOK_OFF_SINT_PM : HOOK_OFF_SINT_RM;
      break;
    case HOOK_HINT:
      off = isPM ? HOOK_OFF_HINT_PM : HOOK_OFF_HINT_RM;
      break;
    case HOOK_EXCEP:
    case HOOK_EXCEP_NO:
    default:
      off = isPM ? HOOK_OFF_EXCEP_PM : HOOK_OFF_EXCEP_RM;
      break;
    }
    if(BX_CPU_THIS_PTR inthook[addr][off].enable) {
      trapfunc = BX_CPU_THIS_PTR inthook[addr][off].trapfunc;
      this_ptr = BX_CPU_THIS_PTR inthook[addr][off].this_ptr;
      goto found;
    }
    return 0;
  default:
  illhook:
    BX_PANIC(("cpu:illegal bios hook addr=%.8x, type=%d.\n",
	      addr,
	      type));
  }

found:
  BX_CPU_THIS_PTR inbios = 1;
  leasttime = 0;
  maxtime = (Bit64u)-1;
  start_time = bx_pc_system.get_current_time();
  trapfunc(this_ptr,
	   this,
	   type,
	   addr,
	   errorno,
	   isPM,
	   exittype,
	   exitaddr,
	   exitseg,
	   leasttime,
	   maxtime);
  time = bx_pc_system.get_current_time() - start_time;
  if(time < leasttime) {
    time = leasttime;
  }
  if(time > maxtime) {
    time = maxtime;
  }
  bx_pc_system.tickn(time * bx_options.ips / 1000000);

  switch(exittype) {
  case HOOK_END_DONE:
    BX_CPU_THIS_PTR inbios = 0;
    return 1;
  case HOOK_END_ORIG:
    BX_CPU_THIS_PTR inbios = 0;
    return 0;
  case HOOK_END_FRET:
    ix.os_32 = BX_CPU_THIS_PTR sregs[BX_SEG_REG_CS].cache.u.segment.d_b;
    if(ix.os_32) {
      RETfar32(&ix);
    } else {
      RETfar16(&ix);
    }
    BX_CPU_THIS_PTR inbios = 0;
    return 1;
  case HOOK_END_NRET:
  case HOOK_END_FJMP:
  case HOOK_END_NJMP:
  case HOOK_END_FCALL:
  case HOOK_END_NCALL:
  case HOOK_END_SINT:
  case HOOK_END_EXP:
  case HOOK_END_EXN:
  case HOOK_END_FLUSH:
  case HOOK_END_PEND:
    BX_PANIC(("cpu:not implemented bios hook addr=%.8x, type=%d, exittype=%d.\n",
	      addr,
	      type,
	      exittype));
  default:
    BX_PANIC(("cpu:illegal bios hook addr=%.8x, type=%d, exittype=%d.\n",
	      addr,
	      type,
	      exittype));
  }

  return 0;
}
#endif
